//*************************************************************************************************
//
//	Description:
//		windows.fx - Specialisation of the basic object shader which supports time-of-day variation
//								 for building windows.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     04/27/2006  0.1           Created
//		TNettleship     05/12/2006  1.0           First "offical" release to FTP
//		TNettleship     05/18/2006  1.1           Added technique annotations.
//		TNettleship     06/09/2006  1.2           Added ambient occlusion term from vertex colour,
//																							global emap factor, removed ambient light, added
//																							option for compressed normals.
//		TNettleship			09/01/2006	1.3						Rewrote the window lighting function.
//		TNettleship			10/03/2006	1.4						Fixed a bug with channel assignment & vertex colours
//																							in 3DSMAX.
//		TNettleship			10/18/2006	1.5						Changed the way lighting distance attenuation works.
//		TNettleship			10/19/2006	1.6						Removed texture mapping info params from display,
//																							reverted to parser 0x0000 until the texture management
//																							bugs in 0x0001 can be fixed.
//		TNettleship			10/25/2006	1.7						Changed binormals from a calculation back to an input.
//		TNettleship			11/02/2006	1.8						Fixed a bug with directional lighting.
//		TMann						11/23/2006	1.10					Added _WIN_GL_ support
//		TMann						11/27/2006	1.11					Texture filtering for GL
//		TMann						12/11/2006	1.13					PS3 version
//		TNettleship			02/01/2006	1.14					Pixel shader optimisations
//		TNettleship			02/12/2007	1.15					Fixed environment mapping
//		TNettleship			04/19/2007	2.0						Added shader specialisation.
//		TNettleship			07/11/2007	2.01					Changed lighting to work in world coords.
//		TNettleship     07/24/2007  2.02          Made sure samplers aren't using anisotropic filtering.
//		TNettleship     08/17/2007  2.03					Removed texture bias.
//		TNettleship			10/23/2007	2.04					Converted to on-load rendermode behaviour binding.
//	<TABLE>
//
//*************************************************************************************************

#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"



//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define MAX_ENVMAP_BIAS	6.0f



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif


//
// Transforms
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
float4x4 worldI : WorldInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;
#endif



//
// Time control
//

#ifdef _3DSMAX_

// Time of day needs to be seperated for 3dsmax UI
float timeOfDayHours
<
	string UIName = "Time of day (hours)";
	float UIMin = 0.0f;
	float UIMax = 23.0f;
	float UIStep = 1.0f;
	bool export = false;
> = 0;

float timeOfDayMins
<
	string UIName = "Time of day (mins)";
	float UIMin = 0.0f;
	float UIMax = 59.0f;
	float UIStep = 1.0f;
	bool export = false;
> = 0;

#else

// In the engine it is combined and in the correct format
float timeOfDay : timeOfDay
<
	bool appEdit = false;
> = 0.0f;

#endif

//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = 2;
	int RuntimeTexcoord = 1;
	bool export = false;
> = 0;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, window effect scalar in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, window effect scalar in alpha
#endif
<
	string UIName = "Diffuse Texture";
	bool appEdit = true;
>;

texture emissiveTexture : TEXTURE							// Window lighting emissive colour texture
<
	string UIName = "Emissive Texture";
	bool appEdit = true;
>;

texture specularTexture : TEXTURE							// Specular colour in RGB, shininess in alpha
<
	string UIName = "Specular Texture";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting
DECLARE_DEPENDENT_VERTEX_STREAM( tangentDependency, tangent, TANGENT, useNormalMap )
DECLARE_DEPENDENT_VERTEX_STREAM( binormalDependency, binormal, BINORMAL, useNormalMap )

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Texture", useNormalMap )
#endif

texture lightingControlTexture : TEXTURE			// Window lighting control texture (time control in RG)
<
	string UIName = "Windows Texture";
	bool appEdit = true;
>;

SPECIALISATION_PARAM( useEnvironmentMap, "Use environment map?", "ENV_MAPPING" )	// TRUE if the environment map is to be used

#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
DEPENDENT_CUBE_TEXTURE_PARAM( environmentTexture, "Env Texture", useEnvironmentMap, false )
#endif

float minSpecPower
<
	string UIName = "Min Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Specular Power";
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
	bool appEdit = true;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

float globalEMapFactor
<
	string UIName = "EMap Factor";
	float UIMin = 0.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 1.0f;

//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D lightingControlMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="lightingControlTexture"; 
	string MinFilter = "Point";
	string MagFilter = "Point";
	string MipFilter = "None";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < lightingControlTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D emissiveMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="emissiveTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < emissiveTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
samplerCUBE environmentMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="environmentTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < environmentTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


//-----------------------------------------------------------------------
//
// Functions
//

//-----------------------------------------------------------------------
//
// CalculateWindowLighting - Calculates whether the current pixel should be lit from
//							 within by window light. This is based on whether the time
//							 of day is within the on/off times passed.
//
// Parameters:
//	_onOffTimes : times of day the window light switches on/off (RG and BA paired)
//
// Return:
//	Amount of lighting to apply.
//
//-----------------------------------------------------------------------

float CalculateWindowLighting( float4 _onOffTimes )
{
#ifdef _3DSMAX_
	// Calculate the current time in the same format as the times passed
	float currentTime = ( timeOfDayHours * 10.0f ) + floor( timeOfDayMins / 6.0f );
#else
	// Engine: the current time of day is in the correct format already
	float currentTime = timeOfDay;
#endif

	//
	// Check onoff pair 1
	//

	float T = currentTime;
	float S = _onOffTimes.r;
	float E = _onOffTimes.g;
	float value = saturate( abs( sign( T - S ) + sign( T - E ) ) );

	if ( S <= E )
	{
		value = 1.0f - value;
	}
	
	//
	// Check onoff pair 2
	//

	S = _onOffTimes.b;
	E = _onOffTimes.a;
	float value2 = saturate( abs( sign( T - S ) + sign( T - E ) ) );

	if ( S <= E )
	{
		value2 = 1.0f - value2;
	}

	return saturate( value + value2 );
}



//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position  : POSITION;													// Object space position
#ifdef _3DSMAX_
	float4 colour    : TEXCOORD1;													// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float2 texCoord1 : TEXCOORD2;													// UV channel 2 texture coord
#else
	float4 colour    : COLOR0;														// Vertex colour
	float2 texCoord0 : TEXCOORD0;													// UV channel 1 texture coord
	float2 texCoord1 : TEXCOORD1;													// UV channel 2 texture coord
#endif
	float3 normal    : NORMAL;														// Object space normal

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent   : TANGENT;														// Object space tangent
	float3 binormal	 : BINORMAL;													// Object space binormal
#endif
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
	float3 normal			: TEXCOORD2;												// Normal vector (world space)
	float4 eye				: TEXCOORD3;												// Eye vector (world space)

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_VS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_VS( 5 )
#endif
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//
// TODO: rearrange to transform eye and lights into local space before
// vector calculations.

VSOUTPUT WindowsVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.colour = _input.colour;
	_output.texCoord0 = _input.texCoord0;
	_output.texCoord1 = _input.texCoord1;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate object-space coordinate frame
	float3 normal  = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	_output.normal = normal;
		DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		_output.tangent  = mul( float4( _input.tangent, 0.0f ), world ).xyz;
		_output.binormal = mul( float4( _input.binormal, 0.0f ), world ).xyz;
#endif
		DEPENDENT_CODE_END( useNormalMap )

	// Calculate object-space vector to the eye
#ifdef _3DSMAX_
// Uncomment if we return to parser 0x0001
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
	float3 normal			: TEXCOORD2;												// Normal vector (world space)
	float4 eye				: TEXCOORD3;												// Eye vector (world space)
	float3 tangent		: TEXCOORD5;												// Tangent vector (world space)
	float3 binormal		: TEXCOORD6;												// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
};

#else

struct PSINPUT
{
	float4 colour			: TEXCOORD4;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
	float3 normal			: TEXCOORD2_centroid;								// Normal vector (world space)
	float4 eye				: TEXCOORD3_centroid;								// Eye vector (world space)

#if defined( NORMAL_MAPPING )
	// These two vertex streams aren't needed without normal_mapping
	float3 tangent		: TEXCOORD5_centroid;								// Tangent vector (world space)
	float3 binormal		: TEXCOORD6_centroid;								// Binormal vector (world space)

	DECLARE_LIGHTING_INTERPOLATORS_PS( 7 )
#else
	DECLARE_LIGHTING_INTERPOLATORS_PS( 5 )
#endif
	DECLARE_SHADOW_PS_INPUTS
};

#endif


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT WindowsFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )
	
	float globalSpecularFactorValue = globalSpecularFactor;
	float minSpecPowerValue = minSpecPower;
	float maxSpecPowerValue = maxSpecPower;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord0 );
	float4 specularTexColour = tex2D( specularMap, _input.texCoord0 );
	float4 lightingControlValues = floor( tex2D( lightingControlMap, _input.texCoord1 ) * 255.0f );
	float4 emissiveColour = tex2D( emissiveMap, _input.texCoord0 );

  // Create tangent space vectors
	float3 TSnormal = normalize( _input.normal );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		float3 tangent = normalize( _input.tangent );
		float3 binormal = normalize( _input.binormal );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord0 );
		float3 normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal and constant specular strength
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Add in window lighting
	accumulatedColour.rgb += CalculateWindowLighting( lightingControlValues ) * emissiveColour.rgb * diffuseTexColour.a;

	// If environment mapping is switched on
	DEPENDENT_CODE_START( useEnvironmentMap )
#if defined( _3DSMAX_ ) || defined( ENV_MAPPING )
		// Calculate the reflection vector
		float3 objectSpaceNormal = mul( float4( normal, 0.0f ), worldI ).xyz;
		float3 objectSpaceEye = mul( float4( -eye, 0.0f ), worldI ).xyz;
		float3 reflectionVector = reflect( objectSpaceEye, objectSpaceNormal );

		// Fetch the environment map colour using the world coords vector
#ifdef _3DSMAX_
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector.xzy, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#else
		float4 environmentTexColour = texCUBElod( environmentMap, float4( reflectionVector, MAX_ENVMAP_BIAS * ( 1.0f - specularTexColour.a ) ) );
#endif

		// Calculate envmap colour (back in tangent space is fine) and add to diffuse
		accumulatedColour += specularTexColour * environmentTexColour * globalEMapFactor;
#endif
	DEPENDENT_CODE_END( useEnvironmentMap )

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateOutputPixel(accumulatedColour);

	return _output;
}




//
// Low Detail Shaders
//



struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD2;														// Vertex colour
	float2 texCoord0	: TEXCOORD0;												// UV coords for texture channel 0
	float2 texCoord1	: TEXCOORD1;												// UV coords for texture channel 1
};



VSOUTPUT_LD WindowsLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
	_output.texCoord0 = _input.texCoord0;
	_output.texCoord1 = _input.texCoord1;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// get normal in world space and do lighting
	float3 normal = normalize( mul( float4( _input.normal, 0.0f ), world ).xyz );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	_output.colour = _input.colour;
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	return _output;
}

PSOUTPUT WindowsLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord0 );
	float4 lightingControlValues = floor( tex2D( lightingControlMap, _input.texCoord1 ) * 255.0f );
	float4 emissiveColour = tex2D( emissiveMap, _input.texCoord0 );

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Add in window lighting
	accumulatedColour.rgb += CalculateWindowLighting( lightingControlValues ) * emissiveColour.rgb * diffuseTexColour.a;


	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateLowDetailOutputPixel(accumulatedColour);

	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Windows
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Windows";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Windows_LowDetail";
	int    lowDetailDeferredID		= 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx WindowsVertexShader();
		PixelShader = compile sce_fp_rsx WindowsFragmentShader();
#else
		VertexShader = compile vs_3_0 WindowsVertexShader();
		PixelShader = compile ps_3_0 WindowsFragmentShader();
#endif		
	}
}

technique Windows_LowDetail
<
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Windows_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx WindowsLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx WindowsLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 WindowsLowDetailVertexShader();
		PixelShader = compile ps_3_0 WindowsLowDetailFragmentShader();
#endif		
	}
}
